//////////////////////////////////////////////////////////////////////////////////////
// ML.h - Fang2MeshLib management class
//
// Author: John Lafleur
//////////////////////////////////////////////////////////////////////////////////////
// THIS CODE IS PROPRIETARY PROPERTY OF SWINGIN' APE STUDIOS, INC.
// Copyright (c) 2002
//
// The contents of this file may not be disclosed to third
// parties, copied or duplicated in any form, in whole or in part,
// without the prior written permission of Swingin' Ape Studios, Inc.
//////////////////////////////////////////////////////////////////////////////////////
// Modification History:
//
// Date     Who         Description
// -------- ----------  --------------------------------------------------------------
// 08/15/02 Lafleur		Created.
//////////////////////////////////////////////////////////////////////////////////////

#ifndef __ML_H_
#define __ML_H_

#include "math.h"
#include "..\ape_file_def.h"
#include "..\KongDef.h"
#include "..\PasmDlg.h"


//
// MeshLib error codes
enum
{
	ML_NO_ERROR = 0,
	ML_GENERAL_ERROR,
	ML_OUT_OF_MEMORY,
	ML_NON_COMPLIANT_DATA_SUBMITTED,
	ML_NO_VERTEX_BUFFERS_ESTABLISHED_FOR_EXPORT,
	ML_NO_SEGMENTS_ESTABLISHED_FOR_EXPORT,
	ML_MESH_VERTEX_DATA_EXCEEDED_LIB_BUFFER,
	ML_UNABLE_TO_RESOLVE_VERT_NORMAL,
	ML_DISPLAY_LIST_DATA_BUFFER_OVERRUN,
	ML_DISPLAY_LIST_CONTAINER_BUFFER_OVERRUN,
	ML_LOD_COUNT_IN_AID_DOES_NOT_MATCH_MESH,
};


//
// MeshLib results struct
//
typedef struct
{
	// Error code generated by the mesh lib
	u32		nErrorCode;

	// Primary information needed for saving to file
	u32		nMeshCount;				// Number of meshes in pExportData
	u32		*panMeshOffsets;		// Array of byte offsets into pExportData where each mesh begins
	void	*pExportData;			// Pointer to the mesh data to be saved to file
	u32		nExportDataSize;		// Size of the data pointed to by pExportData

	// MeshLib performance stats

	// Vertex Compression
		u32		nOrigPosCount;
		u32		nOrigNormalCount;
		u32		nOrigBinormCount;
		u32		nOrigTangCount;
		u32		nOrigSTCount;
		u32		nOrigColorCount;
		u32		nOrigWeightCount;

		u32		nPos32Count;
		u32		nPos16Count;
		u32		nNormal16Count;
		u32		nBinorm16Count;
		u32		nTang16Count;
		u32		nPos8Count;
		u32		nST16Count;
		u32		nColorCount;
		u32		nWeightCount;

	// Mesh Counters
		u32		nStripCount;
		u32		nStripTriCount;
		u32		nListTriCount;
		u32		nBoneCount;
		u32		nSegmentCount;
		u32		nTransDescCount;
		u32		nMaterialCount;
		u32		nFShTexInstCount;
		u32		nTexLyrIDCount;
		u32		nMeshLightCount;
		u32		nVBCount;
		u32		nDLCount;

	// Memory consumed in export data
		u32		nFMeshMem;
		u32		nFPlatMeshMem;
		u32		nFGCMeshSkinMem;
		u32		nBoneMem;
		u32		nSkeletonMem;
		u32		nSegmentMem;
		u32		nShaderMem;
		u32		nGeoDefMem;
		u32		nFIndexMem;
		u32		nMaterialMem;
		u32		nTexLyrIDMem;
		u32		nMeshLightMem;			// Memory consumed by the lights
		u32		nVBMem;					// Memory consumed by the vertex buffer structures

		u32		nCollMem;				// Memory consumed by collision data

		u32		nOrigVertAttrMem;		// Memory that would have been consumed without compression and quantization
		u32		nVertAttrMem;			// Memory consumed by the vertex attributes

		u32		nDLMem;					// Memory consumed by the mesh display list(s)
		u32		nDLMemEstimateExcess;	// Excess memory allocated during display list generation

	// Rendering Info
		u32		nVertsRendered;
		u32		nMatrixSwitches;
		u32		nMatrixLoads;
		u32		nDLChanges;
		u32		nGDBeginCalls;

	// Time consumed generating export data
		f32		fTotalTime;				// Total time to process the mesh, in seconds
		f32		fVertAddTime;			// Time consumed adding verts to the mesh
		f32		fCollisionTime;			// Time to create collision data, in seconds
		f32		fCompressTime;			// Time to compress vertex attribute data, in seconds
		f32		fDLTime;				// Time to create DL data, in seconds
		f32		fDataGeneration;		// Time to create export data, in seconds
		f32		fStrippingTime;			// Time to create mesh strips, in seconds

} MLResults;



class CPCVertHash;
class CGCVertHash;
#ifdef _MMI_TARGET_PS2
class CPS2VertHash;
#endif
class MLMesh;
class MLSegment;
class MLMaterial;
class CCompileDlg;

#ifdef _MMI_TARGET_PS2
struct PS2RawMesh;								// KJ
#endif


//
// Class that manages the conversion of meshes.  There should only be one instance 
// of this class instantiated.
//
class CMLManager
{
	friend class MLMesh;
	public:
		MLResults		m_Results;

		CCompileDlg		*m_pDlg;

		// Vertex Hash Tables
		CPCVertHash		*m_pPCVertHash;
		CGCVertHash		*m_pGCVertHash;
#ifdef _MMI_TARGET_PS2
		CPS2VertHash	*m_pPS2VertHash;		// KJ
		PS2RawMesh		*m_FirstPS2RawMesh;		// KJ
#endif

		BOOL			m_bGenerateMeshStrips;
		BOOL			m_bAccumStreamingData;

		BOOL			m_bVertexRadiosityLit; // This flag will be set when the mesh that is being processed is geometry that may be instanced and lit using vertex radiosity

	private:	// VARIABLES
		TargetPlatform_e	m_nPlatformType;

		MLMesh				*m_pFirstMLMesh;
		MLMesh				*m_pLastMLMesh;

		void				*m_pStreamingData;
		u32					m_nStreamingDataBytes;

		// This BOOL ensures that only one instance of CMLManager is ever instantiated
		static BOOL				m_bInstantiated;

	private:	// METHODS
		void FreeMeshes( void );
		void FreeVertHash( void );

	public:
		CMLManager( void );
		~CMLManager( void );

		void ResetTimers( void );

		// Must be called prior to any mesh conversion
		BOOL InitForPlatform( CCompileDlg *pDlg, TargetPlatform_e nPlatformType, BOOL bGenerateMeshStrips );

		// Must be called prior to any mesh conversion
		TargetPlatform_e GetPlatformType( void ) { return m_nPlatformType; }

		// Call to generate export data for all submitted meshes
		MLResults* GenerateExportData( BOOL bGenerateCollisionData = TRUE );

		// 
		MLMesh*		AllocateMLMesh( char *pszName, BOOL bWorld );

		// 
		void		RemoveFromExportList( MLMesh *pMesh );

		void		AccumStreamingData( void );
		void		FreeStreamingData( void );
		void*		GetStreamingData( u32 *pnDataSize ) { *pnDataSize = m_nStreamingDataBytes; return m_pStreamingData; }
		void		AppendToStreamingData( u32 nDataSize, void **pStartAddress, void **pBaseAddress );
};


// The only instance of the ML Manager
extern CMLManager MLManager;





#endif